using System;

using sharpallegro5;

using ALLEGRO_FONT = System.IntPtr;
using ALLEGRO_EVENT_QUEUE = System.IntPtr;
using ALLEGRO_BITMAP = System.IntPtr;
using ALLEGRO_DISPLAY = System.IntPtr;
using ALLEGRO_TIMER = System.IntPtr;
using ALLEGRO_CONFIG = System.IntPtr;
using ALLEGRO_LOCKED_REGION = System.IntPtr;

/* Tests some drawing primitives.
 */
public class ex_draw : AllegroExample
{
    unsafe struct Example
    {
        public ALLEGRO_FONT font;
        public ALLEGRO_EVENT_QUEUE queue;
        public ALLEGRO_COLOR background, text, white, foreground;
        public ALLEGRO_COLOR outline;
        public ALLEGRO_BITMAP pattern;
        public ALLEGRO_BITMAP zoom;

        public fixed double timer[4], counter[4];
        public int FPS;
        public float text_x, text_y;

        public bool software;
        public int samples;
        public int what;
        public int thickness;
    };
    static Example ex = new Example();

    static string[] names = {
       "filled rectangles",
       "rectangles",
       "filled circles",
       "circles",
       "lines"
    };

    static void draw_pattern(ALLEGRO_BITMAP bmp)
    {
        int w = al_get_bitmap_width(bmp);
        int h = al_get_bitmap_height(bmp);
        int x, y;
        int format = ALLEGRO_PIXEL_FORMAT_BGR_888;
        ALLEGRO_COLOR light = al_map_rgb_f(1, 1, 1);
        ALLEGRO_COLOR dark = al_map_rgb_f(1, 0.9f, 0.8f);
        ALLEGRO_LOCKED_REGION @lock;
        @lock = al_lock_bitmap(bmp, format, ALLEGRO_LOCK_WRITEONLY);
        for (y = 0; y < h; y++)
        {
            for (x = 0; x < w; x++)
            {
                ALLEGRO_COLOR c = ((x + y) & 1) > 1 ? light : dark;
                byte r = 0, g = 0, b = 0;
                IntPtr data = @lock.data;
                al_unmap_rgb(c, ref r, ref g, ref b);
                //data += y * @lock.pitch;
                //data += x * 3;
                //data[0] = r;
                //data[1] = g;
                //data[2] = b;
                al_draw_pixel(x * 3, y * @lock.pitch, c);
            }
        }
        al_unlock_bitmap(bmp);
    }

    static void set_xy(float x, float y)
    {
        ex.text_x = x;
        ex.text_y = y;
    }

    static void print(string format, params object[] arg)
    {
        string message;
        ALLEGRO_STATE state;
        int th = al_get_font_line_height(ex.font);
        al_store_state(ref state, ALLEGRO_STATE_BLENDER);

        message = string.Format(format, arg);

        al_set_blender(ALLEGRO_ADD, ALLEGRO_ONE, ALLEGRO_INVERSE_ALPHA);
        al_draw_textf(ex.font, ex.text, ex.text_x, ex.text_y, 0, "%s", __arglist(message));
        al_restore_state(ref state);

        ex.text_y += th;
    }

    static void primitive(float l, float t, float r, float b,
       ALLEGRO_COLOR color, bool never_fill)
    {
        float cx = (l + r) / 2;
        float cy = (t + b) / 2;
        float rx = (r - l) / 2;
        float ry = (b - t) / 2;
        int tk = never_fill ? 0 : ex.thickness;
        int w = ex.what;
        if (w == 0 && never_fill) w = 1;
        if (w == 2 && never_fill) w = 3;
        if (w == 0) al_draw_filled_rectangle(l, t, r, b, color);
        if (w == 1) al_draw_rectangle(l, t, r, b, color, tk);
        if (w == 2) al_draw_filled_ellipse(cx, cy, rx, ry, color);
        if (w == 3) al_draw_ellipse(cx, cy, rx, ry, color, tk);
        if (w == 4) al_draw_line(l, t, r, b, color, tk);
    }

    static void draw()
    {
        float x, y;
        int cx = 0, cy = 0, cw = 0, ch = 0;
        int w = al_get_bitmap_width(ex.zoom);
        int h = al_get_bitmap_height(ex.zoom);
        ALLEGRO_BITMAP screen = al_get_target_bitmap();
        ALLEGRO_BITMAP mem;
        int rects_num = 16, i, j;
        float[] rects = new float[16 * 4];
        for (j = 0; j < 4; j++)
        {
            for (i = 0; i < 4; i++)
            {
                rects[(j * 4 + i) * 4 + 0] = 2 + i * 0.25f + i * 7;
                rects[(j * 4 + i) * 4 + 1] = 2 + j * 0.25f + j * 7;
                rects[(j * 4 + i) * 4 + 2] = 2 + i * 0.25f + i * 7 + 5;
                rects[(j * 4 + i) * 4 + 3] = 2 + j * 0.25f + j * 7 + 5;
            }
        }

        al_get_clipping_rectangle(ref cx, ref cy, ref cw, ref ch);
        al_clear_to_color(ex.background);

        set_xy(8, 0);
        print("Drawing {0} (press SPACE to change)", names[ex.what]);

        set_xy(8, 16);
        print("Original");

        set_xy(80, 16);
        print("Enlarged x 16");

        al_set_blender(ALLEGRO_ADD, ALLEGRO_ONE, ALLEGRO_ZERO);

        if (ex.software)
        {
            al_set_new_bitmap_flags(ALLEGRO_MEMORY_BITMAP);
            al_set_new_bitmap_format(al_get_bitmap_format(al_get_target_bitmap()));
            mem = al_create_bitmap(w, h);
            al_set_target_bitmap(mem);
            x = 0;
            y = 0;
        }
        else
        {
            mem = NULL;
            x = 8;
            y = 40;
        }
        al_draw_bitmap(ex.pattern, x, y, 0);

        /* Draw the test scene. */

        al_set_blender(ALLEGRO_ADD, ALLEGRO_ONE, ALLEGRO_INVERSE_ALPHA);
        for (i = 0; i < rects_num; i++)
        {
            ALLEGRO_COLOR rgba = ex.foreground;
            rgba.a *= 0.5f;
            primitive(
               x + rects[i * 4 + 0],
               y + rects[i * 4 + 1],
               x + rects[i * 4 + 2],
               y + rects[i * 4 + 3],
               rgba, false);
        }

        al_set_blender(ALLEGRO_ADD, ALLEGRO_ONE, ALLEGRO_ZERO);

        if (ex.software)
        {
            al_set_target_bitmap(screen);
            x = 8;
            y = 40;
            al_draw_bitmap(mem, x, y, 0);
            al_destroy_bitmap(mem);
        }

        /* Grab screen contents into our bitmap. */
        al_set_target_bitmap(ex.zoom);
        al_draw_bitmap_region(screen, x, y, w, h, 0, 0, 0);
        al_set_target_bitmap(screen);

        /* Draw it enlarged. */
        x = 80;
        y = 40;
        al_draw_scaled_bitmap(ex.zoom, 0, 0, w, h, x, y, w * 16, h * 16, 0);

        /* Draw outlines. */
        for (i = 0; i < rects_num; i++)
        {
            primitive(
               x + rects[i * 4 + 0] * 16,
               y + rects[i * 4 + 1] * 16,
               x + rects[i * 4 + 2] * 16,
               y + rects[i * 4 + 3] * 16,
               ex.outline, true);
        }

        set_xy(8, 640 - 48);
        print("Thickness: {0} (press T to change)", ex.thickness);
        print("Drawing with: {0} (press S to change)",
           ex.software ? "software" : "hardware");
        print("Supersampling: {0}x (edit ex_draw.ini to change)", ex.samples);

        // FIXME: doesn't work
        //      al_get_display_option(ALLEGRO_SAMPLE_BUFFERS));
    }

    static void tick()
    {
        draw();
        al_flip_display();
    }

    static void run()
    {
        ALLEGRO_EVENT @event = new ALLEGRO_EVENT();
        bool need_draw = true;

        while (true)
        {
            if (need_draw && al_is_event_queue_empty(ex.queue))
            {
                tick();
                need_draw = false;
            }

            al_wait_for_event(ex.queue, ref @event);

            switch (@event.type)
            {
                case ALLEGRO_EVENT_DISPLAY_CLOSE:
                    return;

                case ALLEGRO_EVENT_KEY_DOWN:
                    if (@event.keyboard.keycode == ALLEGRO_KEY_ESCAPE)
                        return;
                    if (@event.keyboard.keycode == ALLEGRO_KEY_SPACE)
                    {
                        ex.what++;
                        if (ex.what == 5) ex.what = 0;
                    }
                    if (@event.keyboard.keycode == ALLEGRO_KEY_S)
                    {
                        ex.software = !ex.software;
                    }
                    if (@event.keyboard.keycode == ALLEGRO_KEY_T)
                    {
                        ex.thickness++;
                        if (ex.thickness == 2) ex.thickness = 0;
                    }
                    break;

                case ALLEGRO_EVENT_TIMER:
                    need_draw = true;
                    break;
            }
        }
    }

    static void init()
    {
        ex.FPS = 60;

        ex.font = al_load_font("data/fixed_font.tga", 0, 0);
        if (ex.font == IntPtr.Zero)
        {
            Console.Write("data/fixed_font.tga not found.\n");
            Environment.Exit(1);
        }
        ex.background = al_color_name("beige");
        ex.foreground = al_color_name("black");
        ex.outline = al_color_name("red");
        ex.text = al_color_name("blue");
        ex.white = al_color_name("white");
        ex.pattern = al_create_bitmap(32, 32);
        ex.zoom = al_create_bitmap(32, 32);
        draw_pattern(ex.pattern);
    }

    static int Main()
    {
        ALLEGRO_DISPLAY display;
        ALLEGRO_TIMER timer;
        ALLEGRO_CONFIG config;
        string value;
        string str;

        if (!al_init())
        {
            Console.Write("Could not init Allegro.\n");
            return 1;
        }

        al_init_primitives_addon();
        al_install_keyboard();
        al_install_mouse();
        al_init_image_addon();
        al_init_font_addon();

        /* Read supersampling info from ex_draw.ini. */
        ex.samples = 0;
        config = al_load_config_file("ex_draw.ini");
        if (config == IntPtr.Zero)
            config = al_create_config();
        value = al_get_config_value(config, "settings", "samples");
        if (value != null)
            ex.samples = int.Parse(value);
        str = string.Format("{0}", ex.samples);
        al_set_config_value(config, "settings", "samples", str);
        al_save_config_file("ex_draw.ini", config);
        al_destroy_config(config);

        if (ex.samples > 0)
        {
            al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_REQUIRE);
            al_set_new_display_option(ALLEGRO_SAMPLES, ex.samples, ALLEGRO_SUGGEST);
        }
        display = al_create_display(640, 640);
        if (display == IntPtr.Zero)
        {
            Console.Write("Error creating display.\n");
            return 1;
        }

        init();

        timer = al_create_timer(1.0 / ex.FPS);

        ex.queue = al_create_event_queue();
        al_register_event_source(ex.queue, al_get_keyboard_event_source());
        al_register_event_source(ex.queue, al_get_mouse_event_source());
        al_register_event_source(ex.queue, al_get_display_event_source(display));
        al_register_event_source(ex.queue, al_get_timer_event_source(timer));

        al_start_timer(timer);
        run();

        al_destroy_event_queue(ex.queue);

        return 0;
    }
}