using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace sharprd
{
    public class RDPTabPage : RDTabPage
    {
        // No shadow is displayed for the cursor.
        const int TS_PERF_DISABLE_CURSOR_SHADOW = 0x00000020;
        // Cursor blinking is disabled.
        const int TS_PERF_DISABLE_CURSORSETTINGS = 0x00000040;
        // Full-window drag is disabled; only the window outline is displayed when the window is moved.
        const int TS_PERF_DISABLE_FULLWINDOWDRAG = 0x00000002;
        // Menu animations are disabled.
        const int TS_PERF_DISABLE_MENUANIMATIONS = 0x00000004;
        // No features are disabled.
        const int TS_PERF_DISABLE_NOTHING = 0x00000000;
        // Themes are disabled.
        const int TS_PERF_DISABLE_THEMING = 0x00000008;
        // Wallpaper on the desktop is not displayed.
        const int TS_PERF_DISABLE_WALLPAPER = 0x00000001;
        const int TS_PERF_ENABLE_FONT_SMOOTHING = 0x00000080;
        const int TS_PERF_ENABLE_DESKTOP_COMPOSITION = 0x00000100;

        private RdpClient rdp = null;

        private AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEventHandler disconnectedEventHandler;
        private AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEventHandler disconnectedForReconnectionEventHandler;

        public override bool Connected
        {
            get
            {
                return rdp.Connected == 1;
            }
        }

        protected override void OnCreateControl()
        {
            rdp = new RdpClient();
            disconnectedEventHandler = new AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEventHandler(OnDisconnected);
            disconnectedForReconnectionEventHandler = new AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEventHandler(OnDisconnectedForReconnection);
            rdp.OnDisconnected += disconnectedEventHandler;
            rdp.OnConnecting += new EventHandler(OnConnecting);
            rdp.OnConnected += new EventHandler(OnConnected);
            this.Controls.Add(rdp);
            rdp.Dock = DockStyle.Fill;
            base.OnCreateControl();
        }

        private void OnConnecting(object sender, EventArgs e)
        {
            connecting = true;
            Notify(this, new NotifyEventArgs(NotifyEventArgs.Actions.Connecting));
        }

        private void OnConnected(object sender, EventArgs e)
        {
            connecting = false;
            Notify(this, new NotifyEventArgs(NotifyEventArgs.Actions.Connected));
        }

        private void OnDisconnected(object sender, AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEvent e)
        {
            connecting = false;
            Notify(this, new NotifyEventArgs(NotifyEventArgs.Actions.Disconnected));
            ((TabControl)this.Parent).TabPages.Remove(this);
        }

        private void OnDisconnectedForReconnection(object sender, AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEvent e)
        {
            connecting = false;
            Notify(this, new NotifyEventArgs(NotifyEventArgs.Actions.Disconnected));

            rdp.Connect();
            rdp.OnDisconnected -= disconnectedForReconnectionEventHandler;
            rdp.OnDisconnected += disconnectedEventHandler;
        }

        public override void Disconnect()
        {
            try
            {
                if (Connected)
                {
                    rdp.Disconnect();
                }
            }
            catch (Exception)
            {
            }
        }

        public RDPTabPage(RDConnection connection)
            : base(connection)
        {
        }

        public override void Connect()
        {
            connecting = true;

            MSTSCLib.IMsRdpClientAdvancedSettings5 settings = rdp.AdvancedSettings5 as MSTSCLib.IMsRdpClientAdvancedSettings5;

            string server = string.Empty;

            if (!string.IsNullOrEmpty(connection.Computer))
            {
                if (connection.Computer.Contains(":"))
                {
                    string[] tokens = connection.Computer.Split(new char[] { ':' });
                    server = tokens[0];
                    int port = 3389;
                    if (int.TryParse(tokens[1], out port))
                    {
                        settings.RDPPort = port;
                    }
                }
                else
                {
                    server = connection.Computer;
                }
            }

            rdp.Server = server;
            rdp.Domain = connection.Domain;
            rdp.UserName = connection.Username;
            if (!string.IsNullOrEmpty(connection.Password))
            {
                settings.ClearTextPassword = Encryption.Decrypt(connection.Password);
            }
            settings.ConnectToServerConsole = connection.Console;

            if (connection.DisplayMode == RDConnection.DisplayModes.Fullscreen)
            {
                rdp.FullScreen = true;
                rdp.DesktopWidth = Screen.PrimaryScreen.Bounds.Width;
                rdp.DesktopHeight = Screen.PrimaryScreen.Bounds.Height;
            }
            else
            {
                settings.SmartSizing = true;
            }
            rdp.ColorDepth = (int)connection.ColorDepth;
            settings.DisplayConnectionBar = connection.DisplayConnectionBar;

            switch (connection.RemoteAudio)
            {
                case RDConnection.RemoteAudioConfiguration.PlayOnThisComputer:
                    settings.AudioRedirectionMode = 0;
                    break;
                case RDConnection.RemoteAudioConfiguration.DoNotPlay:
                    settings.AudioRedirectionMode = 2;
                    break;
                case RDConnection.RemoteAudioConfiguration.PlayOnRemoteComputer:
                    settings.AudioRedirectionMode = 1;
                    break;
            }

            settings.RedirectPrinters = connection.Printers;
            settings.RedirectClipboard = connection.Clipboard;
            settings.RedirectDrives = connection.Drives;

            if (!connection.DesktopBackground) settings.PerformanceFlags |= TS_PERF_DISABLE_WALLPAPER;
            if (connection.FontSmoothing) settings.PerformanceFlags |= TS_PERF_ENABLE_FONT_SMOOTHING;
            if (connection.DesktopComposition) settings.PerformanceFlags |= TS_PERF_ENABLE_DESKTOP_COMPOSITION;
            if (!connection.WindowContents) settings.PerformanceFlags |= TS_PERF_DISABLE_FULLWINDOWDRAG;
            if (!connection.MenuAnimation) settings.PerformanceFlags |= TS_PERF_DISABLE_MENUANIMATIONS;
            if (!connection.VisualStyles) settings.PerformanceFlags |= TS_PERF_DISABLE_THEMING;
            settings.BitmapPersistence = connection.PersistentBitmapCaching ? 1 : 0;

            rdp.Dock = DockStyle.Fill;

            rdp.Connect();

            ((TabControl)this.Parent).SelectTab(this);
        }

        public override void Reconnect()
        {
            if (Connected)
            {
                rdp.OnDisconnected -= disconnectedEventHandler;
                rdp.OnDisconnected += disconnectedForReconnectionEventHandler;
                rdp.Disconnect();
            }
        }
    }
}
