﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using allegro;
using System.IO;
using System.Xml.Serialization;
using System.Xml;
using System.Threading;

namespace sharpmines
{
    public partial class MainWindow : Form
    {
        private enum Status
        {
            Stop,
            Playing,
            GameOver,
            Won,
            Highscore,
            Exit
        }

        private static int TILE_WIDTH = 23;
        private const int ACTION_TRESHOLD = 300;
        private const int FONT_WIDTH = 12;
        private const int FONT_HEIGHT = 16;

        private static int gridWidth = 10;
        private static int gridHeight = 10;

        private static int mines = 10;
        private static int flags = 0;

        private static int time = 0;
        private static int elapsedTime = 0;

        private static Status status = Status.Stop;

        private static Tile[,] minefield;

        private static bool leftClick = false;
        private static bool rightClick = false;

        private static bool showCredits = false;

        private Highscore score;

        public MainWindow()
        {
            InitializeComponent();

            ClientSize = new Size(232, 231 + 51);

            if (InitializeAllegro())
            {
                allegroWorker.DoWork += new DoWorkEventHandler(AllegroCycle);
                allegroWorker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(allegroWorker_RunWorkerCompleted);
                allegroWorker.RunWorkerAsync();
            }

            // Load highscores
            if (File.Exists("highscores.xml"))
            {
                FileStream stream = new FileStream("highscores.xml", FileMode.Open);
                try
                {
                    XmlSerializer serializer = new XmlSerializer(typeof(Highscore[]));
                    if (stream.Length > 0)
                    {
                        XmlReader reader = new XmlTextReader(stream);
                        Highscore[] highscores = (Highscore[])serializer.Deserialize(reader);
                        reader.Close();

                        Highscore.Highscores.AddRange(highscores);
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show(this, "Unable to load highscore\n" + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                finally
                {
                    stream.Close();
                }
            }
        }

        private void MainWindow_FormClosing(object sender, FormClosingEventArgs e)
        {
            status = Status.Exit;

            Thread.Sleep(500);

            Timer.Instance.Dispose();
            Allegro.allegro_exit();

            // Clean highscores
            Highscore.Highscores.RemoveAll(delegate(Highscore highscore)
            {
                return (string.IsNullOrEmpty(highscore.Name) || highscore.Name.Length < 3);
            });

            StreamWriter writer = new StreamWriter("highscores.xml");
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(Highscore[]));
                serializer.Serialize(writer, Highscore.Highscores.ToArray());
            }
            catch (Exception ex)
            {
                MessageBox.Show(this, "Unable to save highscores\n" + ex.ToString(), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                writer.Close();
            }
        }

        void allegroWorker_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            Close();
        }

        private bool InitializeAllegro()
        {
            Allegro.win_set_window(Handle);

            Allegro.allegro_init();

            Allegro.install_timer();
            Allegro.install_keyboard();
            Allegro.install_mouse();

            if (Allegro.set_gfx_mode(Allegro.GFX_AUTODETECT_WINDOWED, ClientSize.Width, ClientSize.Height, 0, 0) != 0)
            {
                Allegro.allegro_message("Unable to install Allegro");
                return false;
            }

            return true;
        }

        private void AllegroCycle(object sender, DoWorkEventArgs e)
        {
            NewGame();

            BITMAP buffer = Allegro.create_bitmap(Allegro.SCREEN_W, Allegro.SCREEN_H);
            BITMAP panel = Allegro.load_bitmap("images/panel.bmp", Allegro.NULL);
            BITMAP baseTile = Allegro.load_bitmap("images/base.bmp", Allegro.NULL);
            BITMAP coverTile = Allegro.load_bitmap("images/cover.bmp", Allegro.NULL);
            BITMAP flagTile = Allegro.load_bitmap("images/flag.bmp", Allegro.NULL);
            BITMAP[] mineTiles = new BITMAP[2];
            mineTiles[0] = Allegro.load_bitmap("images/mine.bmp", Allegro.NULL);
            mineTiles[1] = Allegro.load_bitmap("images/mine2.bmp", Allegro.NULL);
            BITMAP text = Allegro.load_bitmap("images/text.bmp", Allegro.NULL);
            BITMAP smallText = Allegro.load_bitmap("images/smalltext.bmp", Allegro.NULL);
            BITMAP[] numbers = new BITMAP[8];
            numbers[0] = Allegro.load_bitmap("images/1.bmp", Allegro.NULL);
            numbers[1] = Allegro.load_bitmap("images/2.bmp", Allegro.NULL);
            numbers[2] = Allegro.load_bitmap("images/3.bmp", Allegro.NULL);
            numbers[3] = Allegro.load_bitmap("images/4.bmp", Allegro.NULL);
            numbers[4] = Allegro.load_bitmap("images/5.bmp", Allegro.NULL);
            numbers[5] = Allegro.load_bitmap("images/6.bmp", Allegro.NULL);
            numbers[6] = Allegro.load_bitmap("images/7.bmp", Allegro.NULL);
            numbers[7] = Allegro.load_bitmap("images/8.bmp", Allegro.NULL);
            BITMAP highscores = Allegro.load_bitmap("images/highscores.bmp", Allegro.NULL);
            BITMAP credits = Allegro.load_bitmap("images/credits.bmp", Allegro.NULL);

            int lastAction = Timer.Instance.Time;
            int white = Allegro.makecol(255, 255, 255);

            while (!Allegro.key[Allegro.KEY_ESC] && status != Status.Exit)
            {
                if (status == Status.Playing && time != 0)
                {
                    elapsedTime = Timer.Instance.elapsedTime(time) / 1000;
                    if (elapsedTime > 999) elapsedTime = 999;
                }

                if (status != Status.Won && status != Status.Highscore && status != Status.GameOver)
                {
                    bool won = true;
                    for (int j = 0; j < gridHeight; j++)
                    {
                        for (int i = 0; i < gridWidth; i++)
                        {
                            won = (!minefield[i, j].Mine && minefield[i, j].Status) || (minefield[i, j].Mine && minefield[i, j].Flag);

                            if (!won) break;
                        }

                        if (!won) break;
                    }
                    if (won)
                    {
                        status = Status.Won;
                        score = new Highscore(string.Empty, elapsedTime);
                        if (Highscore.IsHighscore(score))
                        {
                            status = Status.Highscore;
                            Highscore.Add(score);
                        }
                    }
                }

                // Reset game
                if (Allegro.key[Allegro.KEY_F2])
                {
                    NewGame();
                }

                if (Timer.Instance.elapsedTime(lastAction) > ACTION_TRESHOLD)
                {
                    if (Allegro.key[Allegro.KEY_F1])
                    {
                        lastAction = Timer.Instance.Time;
                        showCredits = !showCredits;
                    }

                    if (status == Status.Highscore)
                    {
                        if (string.IsNullOrEmpty(score.Name) || score.Name.Length < 3)
                        {
                            for (int i = Allegro.KEY_A; i < Allegro.KEY_Z; i++)
                            {
                                if (Allegro.key[i])
                                {
                                    char c = (char)('A' + (char)(i - Allegro.KEY_A));
                                    score.Name += c;
                                    lastAction = Timer.Instance.Time;
                                }
                            }

                            if (Allegro.key[Allegro.KEY_BACKSPACE] && score.Name.Length > 0)
                            {
                                score.Name = score.Name.Substring(0, score.Name.Length - 1);
                                lastAction = Timer.Instance.Time;
                            }
                        }

                        if (score.Name.Length == 3)
                        {
                            score = null;
                            status = Status.Won;
                        }
                    }
                }

                // Game logic
                if (status == Status.Stop || status == Status.Playing)
                {
                    if ((Allegro.mouse_b & 1) > 0)
                    {
                        if (!leftClick) leftClick = true;
                    }
                    else
                    {
                        if (leftClick)
                        {
                            int x = Allegro.mouse_x / (TILE_WIDTH);
                            int y = Allegro.mouse_y / (TILE_WIDTH);

                            if (x < gridWidth && y < gridHeight)
                            {
                                if (minefield[x, y].Status == false && !minefield[x, y].Flag)
                                {
                                    if (minefield[x, y].Mine) status = Status.GameOver;
                                    else
                                    {
                                        FloodFill8(x, y);
                                    }

                                    if (status == Status.Stop)
                                    {
                                        status = Status.Playing;
                                        time = Timer.Instance.Time;
                                    }
                                }
                            }

                            leftClick = false;
                        }
                    }

                    if ((Allegro.mouse_b & 2) > 0)
                    {
                        if (!rightClick) rightClick = true;
                    }
                    else
                    {
                        if (rightClick)
                        {
                            int x = Allegro.mouse_x / (TILE_WIDTH);
                            int y = Allegro.mouse_y / (TILE_WIDTH);

                            if (x < gridWidth && y < gridHeight)
                            {
                                if (!minefield[x, y].Status)
                                {
                                    if (minefield[x, y].Flag)
                                    {
                                        minefield[x, y].Flag = false;
                                        flags--;
                                    }
                                    else if (!minefield[x, y].Flag && flags < mines)
                                    {
                                        minefield[x, y].Flag = true;
                                        flags++;
                                    }
                                }
                            }

                            rightClick = false;
                        }
                    }
                }

                // Rendering
                Allegro.show_mouse(Allegro.NULL);
                Allegro.clear_bitmap(buffer);

                for (int j = 0; j < gridHeight; j++)
                {
                    for (int i = 0; i < gridWidth; i++)
                    {
                        if (minefield[i, j].Status)
                        {
                            Allegro.draw_sprite(buffer, baseTile, i * TILE_WIDTH, j * TILE_WIDTH);
                        }
                        else
                        {
                            switch (minefield[i, j].Rotation)
                            {
                                case 1:
                                    Allegro.draw_sprite_h_flip(buffer, coverTile, i * TILE_WIDTH, j * TILE_WIDTH);
                                    break;
                                case 2:
                                    Allegro.draw_sprite_v_flip(buffer, coverTile, i * TILE_WIDTH, j * TILE_WIDTH);
                                    break;
                                case 3:
                                    Allegro.draw_sprite_vh_flip(buffer, coverTile, i * TILE_WIDTH, j * TILE_WIDTH);
                                    break;
                                default:
                                    Allegro.draw_sprite(buffer, coverTile, i * TILE_WIDTH, j * TILE_WIDTH);
                                    break;
                            }
                        }

                        if (
                            status == Status.GameOver &&
                            minefield[i, j].Mine
                        )
                        {
                            Allegro.draw_sprite(buffer, mineTiles[(Timer.Instance.Time / 1000) % 2], i * TILE_WIDTH, j * TILE_WIDTH);
                        }

                        if (minefield[i, j].Flag)
                        {
                            Allegro.draw_sprite(buffer, flagTile, i * TILE_WIDTH, j * TILE_WIDTH);
                        }

                        if (minefield[i, j].Status &&
                            !minefield[i, j].Mine
                            )
                        {
                            int boundingMines = BoundingMines(i, j);
                            if (boundingMines > 0)
                            {
                                Allegro.draw_sprite(buffer, numbers[boundingMines - 1], i * TILE_WIDTH, j * TILE_WIDTH);
                            }
                        }
                    }
                }

                Allegro.draw_sprite(buffer, panel, 0, TILE_WIDTH * gridHeight);

                string minesText = string.Format("{0:000}", mines - flags);
                for (int i = 0; i < 3; i++)
                {
                    int digit = minesText.ToCharArray()[i] - 48;
                    Allegro.blit(text, buffer, digit * FONT_WIDTH, 0, 58 + i * FONT_WIDTH, TILE_WIDTH * gridHeight + 7, FONT_WIDTH, FONT_HEIGHT);
                }

                string timeText = string.Format("{0:000}", elapsedTime);
                for (int i = 0; i < 3; i++)
                {
                    int digit = timeText.ToCharArray()[i] - 48;
                    Allegro.blit(text, buffer, digit * FONT_WIDTH, 0, 58 + i * FONT_WIDTH, TILE_WIDTH * gridHeight + 30, FONT_WIDTH, FONT_HEIGHT);
                }

                int textX = Allegro.SCREEN_W - 132 + ((127 - 9 * 12) / 2);
                if (status == Status.GameOver)
                {
                    PrintString(buffer, smallText, "GAME OVER", textX, 251, -1);
                }
                else if (status == Status.Won || status == Status.Highscore)
                {
                    PrintString(buffer, smallText, "GAME WON!", textX, 251, -1);

                    Allegro.draw_sprite(buffer, highscores, 25, 64);

                    for (int i = 0; i < Highscore.Highscores.Count; i++)
                    {
                        Highscore highscore = Highscore.Highscores[i];
                        if (highscore.Name.Length < 3)
                        {
                            if ((Timer.Instance.Time / 500) % 2 == 0)
                            {
                                PrintString(buffer, smallText, string.Format("{0}: {1:000}", highscore.Name.PadRight(3, '-'), highscore.Time), 63, 71 + i * 25, -1);
                            }
                        }
                        else
                        {
                            PrintString(buffer, smallText, string.Format("{0}: {1:000}", highscore.Name, highscore.Time), 63, 71 + i * 25, -1);
                        }
                    }
                }

                if (showCredits)
                {
                    Allegro.draw_sprite(buffer, credits, 25, 64);

                    PrintString(buffer, smallText, "CREDITS:", 63, 71 + 0 * 25, -1);
                    PrintString(buffer, smallText, "Code by\nEugenio Favalli".ToUpper(), 25, 66 + 1 * 25, 11);
                    PrintString(buffer, smallText, "Iron Plague art\nby Daniel Cook\n(Lostgarden.com)".ToUpper(), 25, 78 + 2 * 25, 11);
                }

                Allegro.show_mouse(buffer);
                Allegro.blit(buffer, Allegro.screen, 0, 0, 1, 0, Allegro.SCREEN_W, Allegro.SCREEN_H);
            }
        }

        private static void NewGame()
        {
            Random random = new Random();
            minefield = new Tile[gridWidth, gridHeight];

            for (int j = 0; j < gridHeight; j++)
            {
                for (int i = 0; i < gridWidth; i++)
                {
                    minefield[i, j] = new Tile(i, j);
                    minefield[i, j].Rotation = random.Next(4);
                }
            }

            for (int i = 0; i < mines; i++)
            {
                int x = random.Next(gridWidth);
                int y = random.Next(gridHeight);

                if (minefield[x, y].Mine) i--;

                minefield[x, y].Mine = true;
            }

            status = Status.Stop;

            time = 1;
            elapsedTime = 0;
            flags = 0;
        }

        private static void FloodFill8(int x, int y)
        {
            if (x >= 0 && x < gridWidth && y >= 0 && y < gridHeight &&
            !minefield[x, y].Status && BoundingMines(x, y) == 0)
            {
                minefield[x, y].Status = true;

                FloodFill8(x + 1, y);
                FloodFill8(x - 1, y);
                FloodFill8(x, y + 1);
                FloodFill8(x, y - 1);

                FloodFill8(x - 1, y - 1);
                FloodFill8(x - 1, y + 1);
                FloodFill8(x + 1, y - 1);
                FloodFill8(x + 1, y + 1);
            }
            else if (x >= 0 && x < gridWidth && y >= 0 && y < gridHeight &&
            !minefield[x, y].Status && BoundingMines(x, y) != 0)
            {
                minefield[x, y].Status = true;
            }
        }

        private static int BoundingMines(int x, int y)
        {
            int boundingMines = 0;

            for (int j = -1; j < 2; j++)
            {
                for (int i = -1; i < 2; i++)
                {
                    int temp_x = x + i;
                    int temp_y = y + j;

                    if (temp_x >= 0 && temp_y >= 0 &&
                        temp_x < gridWidth && temp_y < gridHeight
                        && !(temp_x == x && temp_y == y))
                    {
                        if (minefield[temp_x, temp_y].Mine) boundingMines++;
                    }
                }
            }

            return boundingMines;
        }

        private static void PrintString(IntPtr bmp, IntPtr font, string text, int x, int y, int overrideWidth)
        {
            int fontWidth = 12;
            int fontHeight = 12;
            int left = 0;
            int top = 0;

            for (int i = 0; i < text.Length; i++)
            {
                char c = text[i];
                int character = 39; // Question mark
                if (c >= 'A' && c <= 'Z')
                {
                    character = c - 'A';
                }
                else if (c >= '0' && c <= '9')
                {
                    character = (c - '0') + 26;
                }
                else if (c == '.')
                {
                    character = 36;
                }
                else if (c == ':')
                {
                    character = 37;
                }
                else if (c == '!')
                {
                    character = 38; // Exclamation mark
                }
                else if (c == '-')
                {
                    character = 41;
                }
                else if (c == '(')
                {
                    character = 45;
                }
                else if (c == ')')
                {
                    character = 46;
                }
                else if (c == '\n')
                {
                    top++;
                    left = -1;
                }

                if (c != ' ' && c != '\n')
                {
                    Allegro.masked_blit(
                        font, bmp,
                        character * fontWidth, 0,
                        x + left * (overrideWidth > 0 ? overrideWidth : fontWidth), y + top * fontHeight,
                        fontWidth, fontHeight);
                }

                left++;
            }
        }
    }
}
