using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace sharprd
{
    public class RDPTabPage : TabPage
    {
        // No shadow is displayed for the cursor.
        const int TS_PERF_DISABLE_CURSOR_SHADOW = 0x00000020;
        // Cursor blinking is disabled.
        const int TS_PERF_DISABLE_CURSORSETTINGS = 0x00000040;
        // Full-window drag is disabled; only the window outline is displayed when the window is moved.
        const int TS_PERF_DISABLE_FULLWINDOWDRAG = 0x00000002;
        // Menu animations are disabled.
        const int TS_PERF_DISABLE_MENUANIMATIONS = 0x00000004;
        // No features are disabled.
        const int TS_PERF_DISABLE_NOTHING = 0x00000000;
        // Themes are disabled.
        const int TS_PERF_DISABLE_THEMING = 0x00000008;
        // Wallpaper on the desktop is not displayed.
        const int TS_PERF_DISABLE_WALLPAPER = 0x00000001;
        const int TS_PERF_ENABLE_FONT_SMOOTHING = 0x00000080;
        const int TS_PERF_ENABLE_DESKTOP_COMPOSITION = 0x00000100;

        private List<IObserver> observers;

        private RdpConnection connection;
        private AxMSTSCLib.AxMsRdpClient6 rdp = null;

        public bool Connected
        {
            get
            {
                return rdp.Connected == 1;
            }
        }

        protected override void OnCreateControl()
        {
            rdp = new AxMSTSCLib.AxMsRdpClient6();
            rdp.OnDisconnected += new AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEventHandler(rdpc_OnDisconnected);
            rdp.OnConnected += new EventHandler(rdp_OnConnected);
            this.Controls.Add(rdp);
            rdp.Dock = DockStyle.Fill;
            base.OnCreateControl();
        }

        void rdp_OnConnected(object sender, EventArgs e)
        {
            Notify();
        }

        void rdpc_OnDisconnected(object sender, AxMSTSCLib.IMsTscAxEvents_OnDisconnectedEvent e)
        {
            Notify();
            ((TabControl)this.Parent).TabPages.Remove(this);
        }

        public void Disconnect()
        {
            try
            {
                if (Connected)
                {
                    rdp.Disconnect();
                }
            }
            catch (Exception)
            {
            }
        }

        public RDPTabPage(RdpConnection connection)
        {
            observers = new List<IObserver>();
            Attach(connection);
            this.connection = connection;
            Text = connection.Name;
        }

        public void Connect()
        {
            MSTSCLib.IMsRdpClientAdvancedSettings5 settings = rdp.AdvancedSettings5 as MSTSCLib.IMsRdpClientAdvancedSettings5;

            rdp.Server = connection.Computer;
            rdp.Domain = connection.Domain;
            rdp.UserName = connection.Username;
            if (!string.IsNullOrEmpty(connection.Password)) settings.ClearTextPassword = connection.Password;
            settings.ConnectToServerConsole = connection.Console;

            if (connection.DisplayMode == RdpConnection.DisplayModes.Fullscreen)
            {
                rdp.FullScreen = true;
                rdp.DesktopWidth = Screen.PrimaryScreen.Bounds.Width;
                rdp.DesktopHeight = Screen.PrimaryScreen.Bounds.Height;
            }
            else
            {
                settings.SmartSizing = true;
            }
            rdp.ColorDepth = (int)connection.ColorDepth;
            settings.DisplayConnectionBar = connection.DisplayConnectionBar;

            settings.RedirectPrinters = connection.Printers;
            settings.RedirectClipboard = connection.Clipboard;
            settings.RedirectDrives = connection.Drives;

            if (!connection.DesktopBackground) settings.PerformanceFlags |= TS_PERF_DISABLE_WALLPAPER;
            if (connection.FontSmoothing) settings.PerformanceFlags |= TS_PERF_ENABLE_FONT_SMOOTHING;
            if (connection.DesktopComposition) settings.PerformanceFlags |= TS_PERF_ENABLE_DESKTOP_COMPOSITION;
            if (!connection.WindowContents) settings.PerformanceFlags |= TS_PERF_DISABLE_FULLWINDOWDRAG;
            if (!connection.MenuAnimation) settings.PerformanceFlags |= TS_PERF_DISABLE_MENUANIMATIONS;
            if (!connection.VisualStyles) settings.PerformanceFlags |= TS_PERF_DISABLE_THEMING;
            settings.BitmapPersistence = connection.PersistentBitmapCaching ? 1 : 0;

            rdp.Dock = DockStyle.Fill;

            rdp.Connect();
        }

        public void Attach(IObserver observer)
        {
            observers.Add(observer);
        }

        public void Detach(IObserver observer)
        {
            observers.Remove(observer);
        }

        public void Notify()
        {
            foreach (IObserver observer in observers)
            {
                observer.Update();
            }
        }
    }
}
