﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using System.Xml.Serialization;
using System.Xml;

namespace sharpsyntax
{
    public class Language
    {
        private string name;
        private bool caseSensitive;
        private List<MatchGroup> groups;

        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        public bool CaseSensitive
        {
            get { return caseSensitive; }
            set { caseSensitive = value; }
        }

        public List<MatchGroup> Groups
        {
            get { return groups; }
            set { groups = value; }
        }

        public Language()
        {
        }

        public Language(string name, bool caseSensitive)
        {
            this.name = name;
            this.caseSensitive = caseSensitive;

            this.groups = new List<MatchGroup>();
        }

        public MatchCollection Matches(MatchGroup group, string input)
        {
            switch (group.Type)
            {
                case MatchGroup.MatchType.Symbol:
                    return SymbolMatches(input, group.Tokens.ToArray());
                case MatchGroup.MatchType.Word:
                    return WordMatches(input, group.Tokens.ToArray());
                case MatchGroup.MatchType.RegularExpression:
                    return RegularExpressionMatches(input, group.Tokens.ToArray());
            }
            return null;
        }

        private MatchCollection SymbolMatches(string input, string[] symbols)
        {
            StringBuilder pattern = new StringBuilder();
            if (symbols != null)
            {
                pattern.Append("[" + string.Join(string.Empty, symbols) + "]");
            }

            Regex regex = new Regex(pattern.ToString(), caseSensitive ? RegexOptions.None : RegexOptions.IgnoreCase);
            return regex.Matches(input);
        }

        private MatchCollection WordMatches(string input, string[] words)
        {
            StringBuilder pattern = new StringBuilder();
            if (words != null)
            {
                pattern.Append(@"\b(" + string.Join("|", words) + @")\b");
            }

            Regex regex = new Regex(pattern.ToString(), caseSensitive ? RegexOptions.None : RegexOptions.IgnoreCase);
            return regex.Matches(input);
        }

        private MatchCollection RegularExpressionMatches(string input, string[] regularExpressions)
        {
            string[] patterns = new string[regularExpressions.Length];

            for (int i = 0; i < regularExpressions.Length; i++)
            {
                patterns[i] = string.Format("({0})", regularExpressions[i]);
            }

            Regex regex = new Regex(string.Join("|", patterns));
            return regex.Matches(input);
        }

        public void Serialize()
        {
            MemoryStream stream = new MemoryStream();
            XmlSerializer serializer = new XmlSerializer(this.GetType(), new Type[] { new List<MatchGroup>().GetType() } );
            XmlTextWriter xmlWriter = new XmlTextWriter(stream, Encoding.UTF8);
            xmlWriter.Formatting = Formatting.Indented;
            serializer.Serialize(xmlWriter, this);
            stream = (MemoryStream)xmlWriter.BaseStream;
            string str = Encoding.UTF8.GetString(stream.ToArray());
            TextWriter writer = File.CreateText(name.ToLower() + ".xml");
            writer.Write(str);
            writer.Close();
        }

        public static Language Deserialize(string fullName)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(Language));
            TextReader reader = File.OpenText(fullName);
            MemoryStream stream = new MemoryStream(Encoding.UTF8.GetBytes(reader.ReadToEnd()));
            reader.Close();
            XmlTextWriter xmlWriter = new XmlTextWriter(stream, Encoding.UTF8);
            return (Language)serializer.Deserialize(stream);
        }

        public override string ToString()
        {
            return name;
        }
    }
}
