using System;
using System.Diagnostics;
using System.Reflection;

namespace toolbag
{
    abstract public class ApplicationLog
    {
        public enum LogLevel
        {
            Information,
            Warning,
            Error
        }

        [Flags]
        public enum LogProperties
        {
            DateTime = 1,
            Assembly = 2,
            Method = 4,
        }

        private int frameLevel = 6;

        protected LogLevel logLevel = LogLevel.Information;

        public int FrameLevel
        {
            get
            {
                return frameLevel;
            }

            set
            {
                frameLevel = value;
            }
        }

        protected ApplicationLog()
            : this(LogLevel.Information)
        {
        }

        protected ApplicationLog(LogLevel logLevel)
        {
            this.logLevel = logLevel;
        }

        protected string PrepareMessage(string message, LogProperties properties, int frameLevel)
        {
            StackTrace trace = new StackTrace();
            StackFrame frame = trace.GetFrame(frameLevel);
            MethodBase method = frame.GetMethod();
            string parameters = "";

            int length = method.GetParameters().Length;
            for (int i = 0; i < length; i++)
            {
                ParameterInfo parameter = method.GetParameters()[i];
                parameters += parameter.ParameterType.Name + " " + parameter.Name;
                if (i < length - 1)
                {
                    parameters += ", ";
                }
            }

            string output = string.Empty;

            if ((properties & LogProperties.DateTime) > 0)
            {
                output += DateTime.Now.ToString("ddMMyy HHmmss") + "\t";
            }

            if ((properties & LogProperties.Assembly) > 0)
            {
                output += string.Format("[{0}]\t", method.ReflectedType.Name);
            }

            if ((properties & LogProperties.Method) > 0)
            {
                output += string.Format(
                "{0}({1}):\t",
                method.Name,
                parameters);
            }

            output += message;

            return output;
        }

        protected string PrepareMessage(string message)
        {
            return PrepareMessage(message, LogProperties.DateTime | LogProperties.Assembly | LogProperties.Method, frameLevel + 1);
        }

        protected string PrepareMessage(string message, LogProperties properties)
        {
            return PrepareMessage(message, properties, frameLevel + 1);
        }

        abstract protected void WriteLogEntry(string message, LogLevel logLevel);

        public void Write(string message)
        {
            try
            {
                Write(message, LogLevel.Information);
            }
            catch (Exception ex)
            {
                Debug.Write(ex);
            }
        }

        public void Write(string message, LogLevel logLevel)
        {
            try
            {
                if (logLevel >= this.logLevel)
                {
                    WriteLogEntry(message, logLevel);
                }
            }
            catch (Exception ex)
            {
                Debug.Write(ex);
            }
        }

        public void Error(string message)
        {
            try
            {
                WriteLogEntry(message, LogLevel.Error);
            }
            catch (Exception ex)
            {
                Debug.Write(ex);
            }
        }

        public void Exception(Exception exception)
        {
            try
            {
                WriteLogEntry(exception.ToString(), LogLevel.Error);
            }
            catch (Exception ex)
            {
                Debug.Write(ex);
            }
        }
    }
}