using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;

namespace sharpcapture
{
    public partial class MainWindow : Form
    {
        private const float PEN_WIDTH = 2f;
        private Bitmap bmp;

        BufferedGraphicsContext context;
        BufferedGraphics buffer;

        private Point offset;
        private Point? firstPoint;

        // Actions
        private bool dragging;
        private bool capturing;
        private bool editing;

        // Mouse coordinates
        int mouseX;
        int mouseY;

        Form zoomForm;
        Toolbox toolboxForm;

        public MainWindow()
        {
            InitializeComponent();

            InitializaGraphics();

            Clear();
        }

        private void InitializaGraphics()
        {
            context = new BufferedGraphicsContext();
            buffer = context.Allocate(this.CreateGraphics(), this.DisplayRectangle);
        }

        private void CopyToClipboard()
        {
            Clipboard.SetDataObject(bmp, true);
        }

        private void TakeScreenshot()
        {
            Hide();

            bmp = new Bitmap(ClientRectangle.Width, ClientRectangle.Height, PixelFormat.Format32bppArgb);
            Graphics g = Graphics.FromImage(bmp);
            g.CopyFromScreen(
                Location.X + (Width - ClientRectangle.Width) / 2,
                Location.Y + Height - ClientRectangle.Height - (Width - ClientRectangle.Width) / 2,
                0, 0, ClientRectangle.Size, CopyPixelOperation.SourceCopy);

            CopyToClipboard();
            Opacity = 1;

            Show();
        }

        private void SaveScreenshot()
        {
            SaveFileDialog saveScreenshot = new SaveFileDialog();
            if (bmp != null && saveScreenshot.ShowDialog() == DialogResult.OK)
            {
                bmp.Save(saveScreenshot.FileName);
            }
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Exit();
        }

        private void Exit()
        {
            context.Dispose();

            Close();
        }

        private void Clear()
        {
            // Reset actions
            capturing = true;
            editing = false;

            bmp = null;
            Opacity = 0.5;
            firstPoint = null;
            buffer.Graphics.Clear(Color.FromArgb(128, Color.White));

            Cursor = Cursors.Default;
            zoomForm = null;
            HideToolbox();
        }

        private void MainWindow_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Escape)
            {
                Exit();
            }
            else if (e.Control && e.KeyCode == Keys.C)
            {
                TakeScreenshot();
            }
            else if (e.Control && e.KeyCode == Keys.S)
            {
                SaveScreenshot();
            }
            else if (e.Control && e.KeyCode == Keys.M)
            {
                if (toolboxForm == null)
                {
                    ShowToolbox();
                }
                else
                {
                    HideToolbox();
                }
            }
            else if (e.Control && e.KeyCode == Keys.Z)
            {
                if (zoomForm == null)
                {
                    zoomForm = new Form();
                    zoomForm.Text = "Zoom";
                    zoomForm.FormBorderStyle = FormBorderStyle.SizableToolWindow;
                    zoomForm.Width = 100;
                    zoomForm.Height = 100;
                    PictureBox zoomArea = new PictureBox();
                    zoomArea.Name = "zoomArea";
                    zoomArea.Dock = DockStyle.Fill;
                    zoomForm.Controls.Add(zoomArea);
                    zoomForm.FormClosed += new FormClosedEventHandler(zoomForm_FormClosed);
                    zoomForm.Show();
                }
                else
                {
                    zoomForm.Close();
                    zoomForm = null;
                }
            }
            // Pin/unpin
            else if (e.Control && e.KeyCode == Keys.P)
            {
                TopMost = !TopMost;
            }
        }

        void zoomForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            zoomForm = null;
        }

        void toolboxForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            HideToolbox();
        }

        private void RefreshZoomArea(int mouseX, int mouseY)
        {
            if (zoomForm != null && zoomForm.Controls["zoomArea"] != null)
            {
                Bitmap zoomBmp = new Bitmap(50, 50, PixelFormat.Format32bppArgb);
                Graphics g = Graphics.FromImage(zoomBmp);
                g.CopyFromScreen(
                    Location.X + (Width - ClientRectangle.Width) / 2 + mouseX - 25,
                    Location.Y + Height - ClientRectangle.Height - (Width - ClientRectangle.Width) / 2 + mouseY - 25,
                    0, 0, new Size(50, 50), CopyPixelOperation.SourceCopy);
                Bitmap bmp2 = new Bitmap(100, 100, PixelFormat.Format32bppArgb);
                Graphics g2 = Graphics.FromImage(bmp2);
                g2.InterpolationMode = InterpolationMode.HighQualityBicubic;
                g2.DrawImage(zoomBmp,
                    new Rectangle(0, 0, 100, 100),
                    new Rectangle(0, 0, 50, 50),
                    GraphicsUnit.Pixel);
                ((PictureBox)zoomForm.Controls["zoomArea"]).Image = (Image)bmp2;
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            if (bmp != null) buffer.Graphics.DrawImageUnscaled(bmp, 0, 0);

            if (firstPoint.HasValue && toolboxForm != null)
            {
                DrawShape(buffer.Graphics, firstPoint.Value, new Point(mouseX, mouseY), true);
            }

            buffer.Render();
        }

        private void ShowToolbox()
        {
            Cursor = Cursors.Cross;
            toolboxForm = new Toolbox();
            toolboxForm.Show();
            toolboxForm.TopMost = true;
        }

        private void HideToolbox()
        {
            Cursor = Cursors.Default;
            if (toolboxForm != null) toolboxForm.Close();
            toolboxForm = null;
        }

        private void DrawShape(Graphics g, Point p1, Point p2, bool temp)
        {
            Color color = temp ? Color.FromArgb(128, toolboxForm.SelectedColor) : toolboxForm.SelectedColor;
            Pen pen = new Pen(color, PEN_WIDTH);

            int x = Math.Min(p1.X, p2.X);
            int y = Math.Min(p1.Y, p2.Y);
            int width = Math.Abs(p1.X - p2.X);
            int height = Math.Abs(p1.Y - p2.Y);

            if (!temp) firstPoint = null;

            switch (toolboxForm.SelectedTool)
            {
                case Toolbox.Tool.Line:
                    g.DrawLine(pen, p1, p2);
                    break;
                case Toolbox.Tool.Path:
                    g.DrawLine(pen, p1, p2);
                    if (!temp) firstPoint = p2;
                    break;
                case Toolbox.Tool.Rectangle:
                    g.DrawRectangle(pen, new Rectangle(x, y, width, height));
                    break;
                case Toolbox.Tool.Ellipse:
                    g.DrawEllipse(pen, new Rectangle(x, y, width, height));
                    break;              
                default:
                    // NOOP
                    break;
            }
        }

        private void MainWindow_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                if (capturing && !editing)
                {
                    TakeScreenshot();
                    capturing = false;
                    ShowToolbox();
                }
                else
                {
                    if (firstPoint == null)
                    {
                        firstPoint = new Point(e.X, e.Y);
                    }
                    else
                    {
                        Graphics g = Graphics.FromImage(bmp);
                        DrawShape(g, firstPoint.Value, new Point(e.X, e.Y), false);
                        Invalidate();
                        CopyToClipboard();
                    }
                }
            }
            else if (e.Button == MouseButtons.Middle)
            {
                dragging = true;
                offset = new Point(e.X, e.Y);
            }
            if (e.Button == MouseButtons.Right)
            {
                if (firstPoint != null)
                {
                    firstPoint = null;
                    Invalidate();
                }
                else
                {
                    Clear();
                }
            }
        }

        private void MainWindow_MouseMove(object sender, MouseEventArgs e)
        {
            mouseX = e.X;
            mouseY = e.Y;

            if (dragging)
            {
                Location = new Point(Left - (offset.X - e.X), Top - (offset.Y - e.Y));
            }

            if (zoomForm != null)
            {
                RefreshZoomArea(e.X, e.Y);
            }

            if (firstPoint != null) Invalidate();
        }

        private void MainWindow_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Middle)
            {
                dragging = false;
            }
        }

        protected override void OnPaintBackground(PaintEventArgs e)
        {
            // NOOP
            //base.OnPaintBackground(e);
        }

        private void MainWindow_Resize(object sender, EventArgs e)
        {
            InitializaGraphics();
            Clear();
        }
    }
}