using System;
using System.Text;

using allegro;
using System.Runtime.InteropServices;

namespace excamera
{
    class excamera : Allegro
    {
#if! M_PI
        const float M_PI = 3.14159F;
#endif


        /* display a nice 8x8 chessboard grid */
        const int GRID_SIZE = 8;


        /* convert radians to degrees */
        static float DEG(float n)
        {
            return ((n) * 180.0F / M_PI);
        }


        /* how many times per second the fps will be checked */
        const int FPS_INT = 2;


        /* uncomment to disable waiting for vsync */
        /* #define DISABLE_VSYNC */


        /* parameters controlling the camera and projection state */
        static int viewport_w = 320;
        static int viewport_h = 240;
        static int fov = 48;
        static float aspect = 1.33f;
        static float xpos = 0;
        static float ypos = -2;
        static float zpos = -4;
        static float heading = 0;
        static float pitch = 0;
        static float roll = 0;

        volatile static int fps;
        volatile static int framecount;



        /* render a tile of the grid */
        static void draw_square(BITMAP bmp, ref MATRIX_f camera, int x, int z)
        {
            V3D_f[] _v = new V3D_f[4], _vout = new V3D_f[8], _vtmp = new V3D_f[8];
            V3D_f[] v = new V3D_f[4], vout = new V3D_f[8], vtmp = new V3D_f[8];
            int[] flags = new int[4], _out = new int[8];
            int c, vc = 0;

            for (c = 0; c < 4; c++)
                v[c] = _v[c];

            for (c = 0; c < 8; c++)
            {
                vout[c] = _vout[c];
                vtmp[c] = _vtmp[c];
            }

            /* set up four vertices with the world-space position of the tile */
            v[0].x = x - GRID_SIZE / 2;
            v[0].y = 0;
            v[0].z = z - GRID_SIZE / 2;

            v[1].x = x - GRID_SIZE / 2 + 1;
            v[1].y = 0;
            v[1].z = z - GRID_SIZE / 2;

            v[2].x = x - GRID_SIZE / 2 + 1;
            v[2].y = 0;
            v[2].z = z - GRID_SIZE / 2 + 1;

            v[3].x = x - GRID_SIZE / 2;
            v[3].y = 0;
            v[3].z = z - GRID_SIZE / 2 + 1;

            /* apply the camera matrix, translating world space . view space */
            for (c = 0; c < 4; c++)
            {
                apply_matrix_f(ref camera, v[c].x, v[c].y, v[c].z,
                   ref v[c].x, ref v[c].y, ref v[c].z);

                flags[c] = 0;

                /* set flags if this vertex is off the edge of the screen */
                if (v[c].x < -v[c].z)
                    flags[c] |= 1;
                else if (v[c].x > v[c].z)
                    flags[c] |= 2;

                if (v[c].y < -v[c].z)
                    flags[c] |= 4;
                else if (v[c].y > v[c].z)
                    flags[c] |= 8;

                if (v[c].z < 0.1)
                    flags[c] |= 16;
            }

            /* quit if all vertices are off the same edge of the screen */
            if ((flags[0] & flags[1] & flags[2] & flags[3]) != 0)
                return;

            if ((flags[0] | flags[1] | flags[2] | flags[3]) != 0)
            {
                /* clip if any vertices are off the edge of the screen */
                //vc = clip3d_f(POLYTYPE_FLAT, 0.1f, 0.1f, 4, v,
                //  vout, vtmp, _out);

                //if (vc <= 0)
                //    return;
            }
            else
            {
                /* no need to bother clipping this one */
                vout[0] = v[0];
                vout[1] = v[1];
                vout[2] = v[2];
                vout[3] = v[3];

                vc = 4;
            }

            /* project view space . screen space */
            for (c = 0; c < vc; c++)
                persp_project_f(vout[c].x, vout[c].y, vout[c].z,
                    ref vout[c].x, ref vout[c].y);

            /* set the color */
            vout[0].c = ((x + z) & 1) != 0 ? makecol(0, 255, 0) : makecol(255, 255, 0);

            /* render the polygon */
            vc = 4;
            V3D_f[] t = new V3D_f[8];
            t[0] = new V3D_f(itofix(50), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[1] = new V3D_f(itofix(100), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[2] = new V3D_f(itofix(150), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[3] = new V3D_f(itofix(200), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[4] = new V3D_f(itofix(250), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[5] = new V3D_f(itofix(300), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[6] = new V3D_f(itofix(350), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            t[7] = new V3D_f(itofix(400), itofix(70), itofix(0), 0.0f, 0.0f, makecol(255, 255, 255));
            BITMAP tbmp = create_bitmap(640, 480);
            IntPtr ttt = IntPtrFromStructArray<V3D_f>(vout); //Marshal.AllocCoTaskMem(8 *(5 * sizeof(float) + sizeof(int)));
            ttt = Marshal.ReadIntPtr(ttt);

            //Marshal.StructureToPtr(t[0], ttt, false);
            //polygon3d_f(tbmp, POLYTYPE_FLAT, NULL, 4, ttt); //;vout);//Marshal.UnsafeAddrOfPinnedArrayElement(t, 0));
            //unsafe
            //{
            //  int* vvv = (int*)vout;
            //  polygon3d_f(tbmp, POLYTYPE_FLAT, NULL, vc, &vout);
            //}
        }

        public static IntPtr IntPtrFromStructArray<T>(T[] InputArray) where T : new()
        {
            int size = InputArray.Length;
            T[] resArray = new T[size];
            //IntPtr[] InPointers = new IntPtr[size];
            int dim = IntPtr.Size * size;
            IntPtr rRoot = Marshal.AllocCoTaskMem(Marshal.SizeOf(InputArray[0]) * size);
            for (int i = 0; i < size; i++)
            {
                Marshal.StructureToPtr(InputArray[i], (IntPtr)(rRoot.ToInt32() + i * Marshal.SizeOf(InputArray[i])), false);
            }
            return rRoot;
        }



        /* draw everything */
        static void render(BITMAP bmp)
        {
            MATRIX_f roller = new MATRIX_f(), camera = new MATRIX_f();
            int x, y, w, h;
            float xfront, yfront, zfront;
            float xup = 0, yup = 0, zup = 0;

            /* clear the background */
            clear_to_color(bmp, makecol(255, 255, 255));

            /* set up the viewport region */
            x = (SCREEN_W - viewport_w) / 2;
            y = (SCREEN_H - viewport_h) / 2;
            w = viewport_w;
            h = viewport_h;

            set_projection_viewport(x, y, w, h);
            rect(bmp, x - 1, y - 1, x + w, y + h, makecol(255, 0, 0));
            set_clip_rect(bmp, x, y, x + w - 1, y + h - 1);

            /* calculate the in-front vector */
            xfront = (float)(Math.Sin(heading) * Math.Cos(pitch));
            yfront = (float)Math.Sin(pitch);
            zfront = (float)(Math.Cos(heading) * Math.Cos(pitch));

            /* rotate the up vector around the in-front vector by the roll angle */
            get_vector_rotation_matrix_f(ref roller, xfront, yfront, zfront,
                 (float)(roll * 128.0 / M_PI));
            apply_matrix_f(ref roller, 0, -1, 0, ref xup, ref yup, ref zup);

            /* build the camera matrix */
            get_camera_matrix_f(ref camera,
                    xpos, ypos, zpos,        /* camera position */
                    xfront, yfront, zfront,  /* in-front vector */
                    xup, yup, zup,           /* up vector */
                    fov,                     /* field of view */
                    aspect);                 /* aspect ratio */

            /* draw the grid of squares */
            for (x = 0; x < GRID_SIZE; x++)
                for (y = 0; y < GRID_SIZE; y++)
                    draw_square(bmp, ref camera, x, y);

            /* overlay some text */
            set_clip_rect(bmp, 0, 0, bmp.w, bmp.h);
            textprintf_ex(bmp, font, 0, 0, makecol(0, 0, 0), -1,
              string.Format("Viewport width: {0} (w/W changes)", viewport_w));
            textprintf_ex(bmp, font, 0, 8, makecol(0, 0, 0), -1,
              string.Format("Viewport height: {0} (h/H changes)", viewport_h));
            textprintf_ex(bmp, font, 0, 16, makecol(0, 0, 0), -1,
              string.Format("Field of view: {0} (f/F changes)", fov));
            textprintf_ex(bmp, font, 0, 24, makecol(0, 0, 0), -1,
              string.Format("Aspect ratio: {0:0.00} (a/A changes)", aspect));
            textprintf_ex(bmp, font, 0, 32, makecol(0, 0, 0), -1,
              string.Format("X position: {0:0.00} (x/X changes)", xpos));
            textprintf_ex(bmp, font, 0, 40, makecol(0, 0, 0), -1,
              string.Format("Y position: {0:0.00} (y/Y changes)", ypos));
            textprintf_ex(bmp, font, 0, 48, makecol(0, 0, 0), -1,
              string.Format("Z position: {0:0.00} (z/Z changes)", zpos));
            textprintf_ex(bmp, font, 0, 56, makecol(0, 0, 0), -1,
              string.Format("Heading: {0:0.00} deg (left/right changes)", DEG(heading)));
            textprintf_ex(bmp, font, 0, 64, makecol(0, 0, 0), -1,
              string.Format("Pitch: {0:0.00} deg (pgup/pgdn changes)", DEG(pitch)));
            textprintf_ex(bmp, font, 0, 72, makecol(0, 0, 0), -1,
              string.Format("Roll: {0:0.00} deg (r/R changes)", DEG(roll)));
            textprintf_ex(bmp, font, 0, 80, makecol(0, 0, 0), -1,
              string.Format("Front vector: {0:0.00}, {1:0.00}, {2:0.00}", xfront, yfront, zfront));
            textprintf_ex(bmp, font, 0, 88, makecol(0, 0, 0), -1,
              string.Format("Up vector: {0:0.00}, {1:0.00}, {2:0.00}", xup, yup, zup));
            textprintf_ex(bmp, font, 0, 96, makecol(0, 0, 0), -1,
              string.Format("Frames per second: {0}", fps));
        }



        /* deal with user input */
        static void process_input()
        {
            double frac, iptr;

            poll_keyboard();

            if (key[KEY_W])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                {
                    if (viewport_w < SCREEN_W)
                        viewport_w += 8;
                }
                else
                {
                    if (viewport_w > 16)
                        viewport_w -= 8;
                }
            }

            if (key[KEY_H])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                {
                    if (viewport_h < SCREEN_H)
                        viewport_h += 8;
                }
                else
                {
                    if (viewport_h > 16)
                        viewport_h -= 8;
                }
            }

            if (key[KEY_F])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                {
                    if (fov < 96)
                        fov++;
                }
                else
                {
                    if (fov > 16)
                        fov--;
                }
            }

            if (key[KEY_A])
            {
                frac = (aspect * 10) - (iptr = Math.Floor(aspect * 10));

                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                {
                    if ((frac > 0.59) && (frac < 0.61))
                        aspect += 0.04f;
                    else
                        aspect += 0.03f;
                    if (aspect > 2)
                        aspect = 2;
                }
                else
                {
                    if ((frac > 0.99) || (frac < 0.01))
                        aspect -= 0.04f;
                    else
                        aspect -= 0.03f;
                    if (aspect < .1)
                        aspect = .1f;
                }
            }

            if (key[KEY_X])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                    xpos += 0.05f;
                else
                    xpos -= 0.05f;
            }

            if (key[KEY_Y])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                    ypos += 0.05f;
                else
                    ypos -= 0.05f;
            }

            if (key[KEY_Z])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                    zpos += 0.05f;
                else
                    zpos -= 0.05f;
            }

            if (key[KEY_LEFT])
                heading -= 0.05f;

            if (key[KEY_RIGHT])
                heading += 0.05f;

            if (key[KEY_PGUP])
                if (pitch > -M_PI / 4)
                    pitch -= 0.05f;

            if (key[KEY_PGDN])
                if (pitch < M_PI / 4)
                    pitch += 0.05f;

            if (key[KEY_R])
            {
                if ((key_shifts & KB_SHIFT_FLAG) > 0)
                {
                    if (roll < M_PI / 4)
                        roll += 0.05f;
                }
                else
                {
                    if (roll > -M_PI / 4)
                        roll -= 0.05f;
                }
            }

            if (key[KEY_UP])
            {
                xpos += (float)(Math.Sin(heading) / 4);
                zpos += (float)(Math.Cos(heading) / 4);
            }

            if (key[KEY_DOWN])
            {
                xpos -= (float)(Math.Sin(heading) / 4);
                zpos -= (float)(Math.Cos(heading) / 4);
            }
        }



        static void fps_check()
        {
            fps = framecount * FPS_INT;
            framecount = 0;
        }

        public static TimerHandler t_fps_check = new TimerHandler(fps_check);



        static int Main()
        {
            BITMAP buffer;

            if (allegro_init() != 0)
                return 1;
            install_keyboard();
            install_timer();

            if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0)
            {
                if (set_gfx_mode(GFX_SAFE, 640, 480, 0, 0) != 0)
                {
                    set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
                    allegro_message(string.Format("Unable to set any graphic mode\n{0}\n",
                        allegro_error));
                    return 1;
                }
            }

            set_palette(desktop_palette);
            buffer = create_bitmap(SCREEN_W, SCREEN_H);
            // TODO: check why it raises a CallbackOnCollectedDelegate exception
            //install_int_ex(fps_check, BPS_TO_TIMER(FPS_INT));

            while (!key[KEY_ESC])
            {
                render(buffer);

#if !DISABLE_VSYNC
                vsync();
#endif

                blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
                framecount++;

                process_input();
            }

            destroy_bitmap(buffer);
            return 0;
        }
    }
}
