﻿using System;

// reference: http://www.killswitchcollective.com/blog/page/6

public class Vector
{
    public double dX;
    public double dY;
    public double x;
    public double y;

    // Default constructor
    public Vector()
    {
    }

    // Origin based vector
    public Vector(double x, double y)
        : this(0, 0, x, y)
    {
    }

    public Vector(double x, double y, double dX, double dY)
    {
        Reset(x, y, dX, dY);
    }

    public void Reset(double x, double y, double dX, double dY)
    {
        this.x = x;
        this.y = y;
        this.dX = dX;
        this.dY = dY;
    }

    public void Reset(Vector v)
    {
        x = v.x;
        y = v.y;
        dX = v.dX;
        dY = v.dY;
    }

    public void SetPolar(double len, double angle)
    {
        Point p = Point.Polar(len, Deg2Rad(angle));
        Reset(x, y, p.X, p.Y);
    }

    public double GetLength()
    {
        return Math.Sqrt(dX * dX + dY * dY);
    }

    public void SetLength(double len)
    {
        Point p = Point.Polar(len, Deg2Rad(GetAngle()));
        Reset(x, y, p.X, p.Y);
    }

    public double GetAngle()
    {
        return Math.Atan2(dY, dX) * 180 / Math.PI;
    }

    public void SetAngle(double angle)
    {
        Point p = Point.Polar(GetLength(), Deg2Rad(angle));
        Reset(x, y, p.X, p.Y);
    }

    public void Rotate(double degrees)
    {
        double len = GetLength();
        double angle = GetAngle();
        dX = len * Math.Cos(Deg2Rad(angle + degrees));
        dY = len * Math.Sin(Deg2Rad(angle + degrees));
    }

    public Vector Unit()
    {
        double len = GetLength();
        if (len == 0) { len = 1; };
        return new Vector(x, y, dX / len, dY / len);
    }

    public Vector RightNormal()
    {
        Vector vR = new Vector(x, y, 0, 0);
        vR.SetPolar(GetLength(), GetAngle() + 90);
        return vR;
    }

    public Vector LeftNormal()
    {
        Vector vL = new Vector(x, y, 0, 0);
        vL.SetPolar(GetLength(), GetAngle() - 90);
        return vL;
    }

    public Point GetStartPoint()
    {
        return new Point(x, y);
    }

    public Point GetEndpoint()
    {
        return new Point(x + dX, y + dY);
    }

    public static Vector operator +(Vector v1, Vector v2)
    {
        return new Vector(v1.x, v1.y, v1.dX + v2.dX, v1.dY + v2.dY);
    }

    public double Dot(Vector v)
    {
        return dX * v.Unit().dX + dY * v.Unit().dY;
    }

    // Projection onto another given vector
    public Vector Project(Vector v)
    {
        double dp = Dot(v);
        return new Vector(v.x, v.y, dp * v.Unit().dX, dp * v.Unit().dY);
    }

    private double perP(Vector va, Vector vb)
    {
        return va.dX * vb.dY - va.dY * vb.dX;
    }

    private double Deg2Rad(double degrees)
    {
        return degrees * Math.PI / 180;
    }

    public Point GetIntersection(Vector v)
    {
        Vector v3 = new Vector(0, 0, v.x - x, v.y - y);
        double t = perP(v3, v) / perP(this, v);
        return new Point(x + dX * t, y + dY * t);
    }
 
    public bool IntersectsWith(Point p)
    {
        Vector startToP = new Vector(x, y, p.X - x, p.Y - y);
        Point endPoint = GetEndpoint();
        Vector endToP = new Vector(endPoint.X, endPoint.Y, p.X - endPoint.X, p.Y - endPoint.Y);
        return (Math.Round(endToP.GetLength() + startToP.GetLength()) == Math.Round(this.GetLength()));
    }

    public Vector ShortestDistance(Vector v)
    {
        Vector v2ToV1 = new Vector(v.x, v.y, x - v.x, y - v.y);
        Vector shortest = v2ToV1.Project(v.LeftNormal());
        shortest.x = x;
        shortest.y = y;
        shortest.dX *= -1;
        shortest.dY *= -1;
        return shortest;
    }

    public override string ToString()
    {
        return ("x: " + x + ", y: " + y + ", deltaX: " + dX + ", deltaY " + dY + ", length: " + GetLength());
    }
}



