using System;
using System.CodeDom.Compiler;
using System.Collections;
using System.Drawing;
using System.Reflection;
using System.Windows.Forms;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Schema;
using System.IO;

namespace Snippeter
{
    public partial class MainWindow : Form
    {
        private Icon errorIcon = new Icon("error-icon.ico");
        private Icon warningIcon = new Icon("warning-icon.ico");

        private bool test = false;

        public MainWindow()
        {
            InitializeComponent();

            snippetBox.Settings.Keywords.AddRange(new string[] { "if", "then", "else" });
            snippetBox.Settings.Keywords.AddRange(new string[] { "sbyte", "byte", "short", "int", "uint", "long", "float", "double", "decimal", "char", "bool", "date", "string" });
            snippetBox.Settings.Keywords.AddRange(new string[] { "String", "Boolean", "Int16", "Int32", "Int64", "DateTime" });

            snippetBox.Settings.Comment = "//";

            snippetBox.Settings.KeywordColor = Color.Blue;
            snippetBox.Settings.CommentColor = Color.Green;
            snippetBox.Settings.StringColor = Color.Gray;
            snippetBox.Settings.IntegerColor = Color.Red;

            snippetBox.CompileKeywords();
        }

        static ArrayList commands = new ArrayList();
        int commandIndex = 0;

        private const string usingDirectives = "using System;using System.IO;using System.Text;";
        private const string classHeader =
            @"public delegate void Proc();
            public class Wrapper {
                public static object Set(string name, object value) {
                    AppDomain.CurrentDomain.SetData(name, value);
                    return value;
                }
                public static object Get(string name) {
                    return AppDomain.CurrentDomain.GetData(name);
                }
                public static object Invoke(Proc proc) {
                    proc();
                    return null;
                }
                public static string Eval() {
                    StringWriter writer = new StringWriter();
                    Console.SetOut(writer);";
        private const string classFooter = "return writer.ToString();}}";


        string StringEval(string expr)
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(usingDirectives);
            stringBuilder.Append(classHeader);
            stringBuilder.Append(Environment.NewLine);
            stringBuilder.Append(expr);
            stringBuilder.Append(Environment.NewLine);
            stringBuilder.Append(classFooter);
            string program = stringBuilder.ToString();

            CodeDomProvider provider = CodeDomProvider.CreateProvider("CSharp");
            CompilerParameters cp = new CompilerParameters();
            cp.GenerateInMemory = true;
            cp.GenerateExecutable = false;

            CompilerResults result = provider.CompileAssemblyFromSource(cp, program);;
            if (result.Errors.HasErrors)
            {
                if (result.Errors[0].ErrorNumber == "CS0029")
                    return StringEval("Invoke(delegate { " + expr + "; })");
                return result.Errors[0].ErrorText;
            }
            else
            {
                Assembly assembly = result.CompiledAssembly;
                Type target = assembly.GetType("Wrapper");
                MethodInfo method = target.GetMethod("Eval");
                object invoke = method.Invoke(null, null);
                return invoke == null ? null : invoke.ToString();
            }
        }

        private void log(string message)
        {
            snippetOutputBox.Text += message + Environment.NewLine;
        }

        private void playButton_Click(object sender, EventArgs e)
        {
            string command = snippetBox.Text.Trim();

            if (command != "")
            {
                try
                {
                    string result = StringEval(command);
                    commands.Add(command);
                    commandIndex = commands.Count;
                    snippetOutputBox.Text = "";
                    log(result);
                }
                catch (TargetInvocationException ex)
                {
                    MessageBox.Show(ex.InnerException.GetType().Name + ": " + ex.InnerException.Message);
                }
                catch (Exception ex)
                {
                    log(ex.GetType().Name + ": " + ex.Message);
                }

            }
        }

        private void textBox2_TextChanged(object sender, EventArgs e)
        {
            // Calculate the starting position of the current line.
            int start = 0, end = 0;

            for (start = snippetBox.SelectionStart - 1; start > 0; start--)
            {
                if (snippetBox.Text[start] == '\n') { start++; break; }
            }
            // Calculate the end position of the current line.
            for (end = snippetBox.SelectionStart; end < snippetBox.Text.Length; end++)
            {
                if (snippetBox.Text[end] == '\n') break;
            }
            // Extract the current line that is being edited.
            String line = snippetBox.Text.Substring(start, end - start);
            // Backup the users current selection point.
            int selectionStart = snippetBox.SelectionStart;
            int selectionLength = snippetBox.SelectionLength;
            // Split the line into tokens.
            Regex r = new Regex("([ \\t{}();])");
            string[] tokens = r.Split(line);
            int index = start;
            foreach (string token in tokens)
            {
                // Set the token's default color and font.
                snippetBox.SelectionStart = index;
                snippetBox.SelectionLength = token.Length;
                snippetBox.SelectionColor = Color.Black;
                snippetBox.SelectionFont = new Font("Courier New", 10,
                FontStyle.Regular);
                // Check whether the token is a keyword.
                String[] keywords = { "public", "void", "using", "static", "class" };
                for (int i = 0; i < keywords.Length; i++)
                {
                    if (keywords[i] == token)
                    {
                        // Apply alternative color and font to highlight keyword.
                        snippetBox.SelectionColor = Color.Blue;
                        snippetBox.SelectionFont = new Font("Courier New", 10, FontStyle.Bold);
                        break;
                    }

                    index += token.Length;
                }
                // Check whether the token is a type.
                String[] types = { "int" };
                for (int i = 0; i < types.Length; i++)
                {
                    if (types[i] == token)
                    {
                        // Apply alternative color and font to highlight keyword.
                        snippetBox.SelectionColor = Color.Red;
                        snippetBox.SelectionFont = new Font("Courier New", 10, FontStyle.Bold);
                        break;
                    }

                    index += token.Length;
                }
                // Restore the users current selection point.
                snippetBox.SelectionStart = selectionStart;
                snippetBox.SelectionLength = selectionLength;
            }
        }

        private void consoleBox_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter && consoleBox.Text.Trim() != "")
            {
                commands.Add(consoleBox.Text);
                consoleOutputBox.Text = StringEval("Console.Write(" + consoleBox.Text + ");");
                consoleBox.Text = "";
            }
            else if (e.KeyCode == Keys.Up)
            {
                if (commands.Count > 0)
                {
                    commandIndex--;
                    if (commandIndex == -1) commandIndex = commands.Count - 1;
                    consoleBox.Text = (string)commands[commandIndex];
                }
            }
        }

        private void regExBox_TextChanged(object sender, EventArgs e)
        {
            try
            {
                validationBox.ForeColor = Color.Green;
                if (Regex.IsMatch(regExBox.Text, validationBox.Text))
                {
                    regExBox.ForeColor = Color.Green;
                }
                else
                {
                    regExBox.ForeColor = Color.Red;
                }
            }
            catch
            {
                validationBox.ForeColor = Color.Red;
            }
        }

        private void tabControl1_SelectedIndexChanged(object sender, EventArgs e)
        {
            playButton.Enabled = tabControl.SelectedIndex == 0;
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void fontsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            FontDialog fontDialog = new FontDialog();
            if (fontDialog.ShowDialog() != DialogResult.Cancel)
            {
                setFont(fontDialog.Font);
            }
        }

        private void setFont(Font font)
        {
            snippetOutputBox.Font = font;
            snippetBox.Font = font;
            consoleBox.Font = font;
            consoleOutputBox.Font = font;
            validationBox.Font = font;
            regExBox.Font = font;
            xmlText.Font = font;
            xmlSchema.Font = font;
        }

        private void xmlSchema_TextChanged(object sender, EventArgs e)
        {
            errorList.Rows.Clear();
            xmlText.ForeColor = Color.Green;
            // No XML, no party!
            if (xmlText.Text == string.Empty) return;

            try
            {
                ValidationEventHandler eventHandler = new ValidationEventHandler(ShowCompileErrors);
                XmlReaderSettings settings = new XmlReaderSettings();
                if (xmlSchema.Text != string.Empty)
                {
                    settings.Schemas.Add(XmlSchema.Read(new StringReader(xmlSchema.Text), eventHandler));
                    settings.ValidationType = ValidationType.Schema;
                }

                XmlDocument document = new XmlDocument();
                document.Load(XmlReader.Create(new StringReader(xmlText.Text), settings));
                if (xmlSchema.Text != string.Empty) document.Validate(eventHandler);
            }
            catch (XmlException ex)
            {
                errorList.Rows.Add(new object[] { errorIcon, "xml", ex.Message });
                xmlText.ForeColor = Color.Red;
            }
            catch (XmlSchemaException ex)
            {
                errorList.Rows.Add(new object[] { errorIcon, "schema", ex.Message });
                xmlText.ForeColor = Color.Red;
            }
            catch (Exception ex)
            {
                errorList.Rows.Add(new object[] { errorIcon, "error", ex.Message });
                xmlText.ForeColor = Color.Red;
            }
            finally
            {
                errorList.CurrentCell = null;
            }
		}

		public void ShowCompileErrors(object sender, ValidationEventArgs args)
		{
            errorList.Rows.Add(new object[] { args.Severity == XmlSeverityType.Error ? errorIcon : warningIcon, "", args.Message });
            xmlText.ForeColor = Color.Red;
		}

        private void tabControl1_MouseHover(object sender, EventArgs e)
        {
            errorList.Visible = true;
            errorList.BringToFront();
        }

        private void errorList_MouseHover(object sender, EventArgs e)
        {
            test = true;
        }

        private void errorList_MouseLeave(object sender, EventArgs e)
        {
            if (test) errorList.Visible = false;
        }
    }
}